#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
arr_make.py — Build an .ARR playlist from pattern list

Input formats:
  1) CSV file: each row = PATTERN[,REPEAT]
     - PATTERN: filename or stem (e.g., RCK_P001 or RCK_P001.ADP)
     - REPEAT: optional int (default 1)
  2) CLI items: PATTERN,REPEAT (comma-separated), e.g., RCK_P001,4 RCK_FILL1,1

Output:
  ; ARR v2.2
  TITLE=...
  DEFAULT_TEMPO=...
  DEFAULT_SWING=...
  SEARCH_PATH=/PATTERNS
  P01=RCK_P001.ADP, REPEAT=4
  P02=RCK_FILL1.ADP, REPEAT=1
  PLAY=P01 P02 P01 ...

Usage:
  python arr_make.py --title "Rock Demo" --tempo 120 --in list.csv --out ./SD/SONGS/ROCK.ARR
  python arr_make.py --title "Blues" --tempo 96 --out ./SD/SONGS/BLUES.ARR RCK_P001,4 RCK_FILL1,1
"""

import argparse, csv, pathlib, sys

def norm_pattern_name(name: str) -> str:
    name = name.strip()
    if not name: return ""
    # append .ADP if no extension
    if "." not in name:
        return name + ".ADP"
    return name

def parse_cli_items(items):
    out = []
    for it in items:
        it = it.strip()
        if not it: continue
        if "," in it:
            pat, rep = it.split(",", 1)
            try:
                repi = int(rep.strip())
            except:
                repi = 1
        else:
            pat, repi = it, 1
        out.append((norm_pattern_name(pat.strip()), max(1, repi)))
    return out

def parse_csv(path: pathlib.Path):
    out = []
    with path.open("r", encoding="utf-8") as f:
        for row in csv.reader(f):
            if not row: continue
            pat = row[0].strip()
            if not pat or pat.startswith("#"):  # comments
                continue
            repi = 1
            if len(row) >= 2 and row[1].strip():
                try:
                    repi = int(row[1].strip())
                except:
                    repi = 1
            out.append((norm_pattern_name(pat), max(1, repi)))
    return out

def main():
    ap = argparse.ArgumentParser(description="Make ARR v2.2 file from patterns list")
    ap.add_argument("--title", required=True, help="Title of the arrangement")
    ap.add_argument("--tempo", type=int, default=120, help="Default tempo (BPM)")
    ap.add_argument("--swing", type=int, default=0, help="Default swing percent (0-100)")
    ap.add_argument("--search-path", default="/PATTERNS", help="Search path used on device")
    ap.add_argument("--in", dest="infile", help="CSV input file (PATTERN[,REPEAT])")
    ap.add_argument("--out", required=True, help="Output .ARR file path")
    ap.add_argument("items", nargs="*", help="Optional CLI items: PATTERN,REPEAT ...")
    args = ap.parse_args()

    items = []
    if args.infile:
        items = parse_csv(pathlib.Path(args.infile))
    if not items and args.items:
        items = parse_cli_items(args.items)

    if not items:
        print("[ERR] no patterns given (use --in CSV or positional 'PATTERN,REPEAT')", file=sys.stderr)
        sys.exit(1)

    # Build sections P01.. and PLAY sequence
    sections = []
    play_tokens = []
    for idx, (pat, rep) in enumerate(items, start=1):
        label = f"P{idx:02d}"
        sections.append((label, pat, rep))
        play_tokens.append(label)

    outp = pathlib.Path(args.out)
    outp.parent.mkdir(parents=True, exist_ok=True)

    with outp.open("w", encoding="utf-8") as w:
        w.write("; ARR v2.2\n")
        w.write(f"TITLE={args.title}\n")
        w.write(f"DEFAULT_TEMPO={max(20,min(300,args.tempo))}\n")
        w.write(f"DEFAULT_SWING={max(0,min(100,args.swing))}\n")
        w.write(f"SEARCH_PATH={args.search_path}\n\n")
        for label, pat, rep in sections:
            w.write(f"{label}={pat}, REPEAT={rep}\n")
        w.write("\n")
        w.write("PLAY=" + " ".join(play_tokens) + "\n")

    print(f"[OK] ARR written: {outp}  (sections={len(sections)})")

if __name__ == "__main__":
    main()
